package eu.javaexperience.japp;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.sonatype.aether.artifact.Artifact;
import org.sonatype.aether.graph.Dependency;

import com.tobedevoured.naether.impl.NaetherImpl;

/**
 * repo
 * package
 * version
 * manifest main/specific class
 * 
 * 
 * 
 * TODO features:
 * 	- export: export application to CLI side (linux use ~/bin by extending $PATH) with custom opts and "$@"
 * 	
 * 	- launch: run a specific version of java application repository:app:version main class ...other opts
 * 	- main: -||- but main class is from the manifest file. 
 * 
 * 	- setup: prompt to select a specific version of package for `run` command.
 * 	- run: run application on a previously selected version and mode (mf, sc)
 * 	
 *  - browse: browse applications on remote maven repository
 *  
 * */
public class JApp
{
	protected static final OpenURLClassLoader CLASS_LOADER = new OpenURLClassLoader(ClassLoader.getSystemClassLoader());
	
	protected static boolean LOG_PREPARATION_TIME = null != System.getenv("JAPP_VERBOSE_TIMES");
	
	protected static long TIME_START = System.currentTimeMillis();
	protected static long TIME_CHECKPOINT;
	
	protected static String getLocalDependencyFile(Dependency dep)
	{
		Artifact art = dep.getArtifact();
		
		return	System.getProperty("user.home")+
				"/.m2/repository/"+
				art.getGroupId().replace('.', '/')+"/"+
				art.getArtifactId()+"/"+
				art.getVersion()+"/"+
				art.getArtifactId()+"-"+art.getVersion()+".jar"
		;
	}
	
	public static void addJarToClassPath(String path) throws MalformedURLException
	{
		CLASS_LOADER.addURL(new File(path).toURI().toURL());
		System.setProperty("java.class.path", System.getProperty("java.class.path")+":"+path);
	}
	
	protected static Dependency addMavenDependencies(String[] deps) throws Exception
	{
		//TODO inject extra dependencies like LD_PRELOAD trick
		NaetherImpl naether = new NaetherImpl();
		
		for(String dep:deps)
		{
			if(occurrenceIn(dep, ":") > 2)
			{
				String[] parts = dep.split(":");
				int l = parts.length;
				dep = parts[l-3]+":"+parts[l-2]+":"+parts[l-1];
				
				naether.addRemoteRepositoryByUrl(join(":", Arrays.copyOf(parts, l-3)));
			}
			
			try
			{
				naether.addDependency(dep);
			}
			catch(Exception e)
			{
				e.printStackTrace();
			}
		}
		
		if(LOG_PREPARATION_TIME)
		{
			System.err.println("JApp before dependencies resolved tooks: "+(System.currentTimeMillis()-TIME_CHECKPOINT)+" ms, total: "+(System.currentTimeMillis()-TIME_START));
		}
		
		try
		{
			naether.resolveDependencies();
		}
		catch(Exception e)
		{
			e.printStackTrace();
		}
		
		if(LOG_PREPARATION_TIME)
		{
			System.err.println("JApp after dependencies resolved tooks: "+(System.currentTimeMillis()-TIME_CHECKPOINT)+" ms, total: "+(System.currentTimeMillis()-TIME_START));
			TIME_CHECKPOINT = System.currentTimeMillis();
		}
		
		Collection<Dependency> ds = naether.currentDependencies();
		Dependency frst = null;
		for(Dependency d:ds)
		{
			if(null == frst)
			{
				frst = d;
			}
			//TODO log System.out.println(d);
			addJarToClassPath(getLocalDependencyFile(d));
		}
		
		if(LOG_PREPARATION_TIME)
		{
			System.err.println("JApp add dependencies ("+ds.size()+"): "+(System.currentTimeMillis()-TIME_CHECKPOINT)+" ms, total: "+(System.currentTimeMillis()-TIME_START));
			TIME_CHECKPOINT = System.currentTimeMillis();
		}
		return frst;
	}
	
	protected static int occurrenceIn(String subject, String search)
	{
		if(search.length() < 1)
		{
			return -1;
		}
		
		int occ = 0;
		int start = 0;
		do
		{
			int index = subject.indexOf(search, start);
			if(index < 0)
			{
				return occ;
			}
			
			++occ;
			start = index+search.length();
		}
		while(true);
	}
	
	protected static String join(String delimiter, String... values)
	{
		StringBuilder sb = new StringBuilder();
		int i=0;
		for(String s:values)
		{
			if(++i > 1)
			{
				sb.append(delimiter);
			}
			sb.append(s);
		}
		
		return sb.toString();
	}
	
	public static void printHelpAndExit()
	{
		System.err.println("JApp maven repository application launcher. Usage:");
		System.err.println("\tlaunch $PACKAGE $MAIN_CLASS ...$ARGS - launch package with main class defined in mainfest file");
		System.err.println("\t\tExample: japp launch org.jmonkeyengine:jme3-examples:3.5.1-stable");
		System.err.println("\tmain $PACKAGE $MAIN_CLASS ...$ARGS - launch package with specified main class");
		System.err.println("\t\tExample: japp main   https://maven.javaexperience.eu/:javaexperience:rpc:1.3.2   eu.javaexperience.rpc.external_lang.ExampleRpcServer");
		System.err.println();
		System.exit(1);
	}
	
	public static void launch(String... args) throws Throwable
	{
		Dependency d = addMavenDependencies(new String[] {args[0]});
		String jar = getLocalDependencyFile(d);
		if(null == jar)
		{
			System.err.println("Can't fetch jar file from dependency: "+d);
			System.exit(1);
		}
		
		String mc  = fetchMainClass(jar);
		
		if(null == mc)
		{
			System.err.println("Can't fetch main class from mainfest file");
			System.exit(1);
		}
		
		Class c = CLASS_LOADER.loadClass(mc);
		Method main = c.getDeclaredMethod("main", String[].class);
		main.invoke(null, new Object[]{Arrays.copyOfRange(args, 1, args.length)});
	}
	
	protected static String fetchMainClass(String jarFile) throws IOException
	{
		String data = null;
		try(ZipFile zf = new ZipFile(jarFile))
		{
			ZipEntry ent = zf.getEntry("META-INF/MANIFEST.MF");
			try(InputStream is = zf.getInputStream(ent))
			{
				data = new String(loadAllFromInputStream(is));
			}
		}
		
		for(String line:data.split("(\r\n|\r|\n)"))
		{
			if(line.startsWith("Main-Class:"))
			{
				return line.substring(11).trim();
			}
		}
		
		return null;
	}
	
	public static byte[] loadAllFromInputStream(InputStream is) throws IOException
	{
		int ep = 0;
		int read = 0;
		byte[] ret = new byte[4096];
		
		while((read = is.read(ret, ep, ret.length-ep))>0)
		{
			if(ep + read == ret.length)
				ret = Arrays.copyOf(ret, ret.length*2);
			
			ep+= read;
		}

		return Arrays.copyOf(ret, ep);
	}
	
	public static void launchMain(String... args) throws Exception
	{
		//TODO check args
		
		//dependencies args
		addMavenDependencies(new String[] {args[0]});
		Class c = CLASS_LOADER.loadClass(args[1]);
		Method main = c.getDeclaredMethod("main", String[].class);
		main.invoke(null, new Object[]{Arrays.copyOfRange(args, 2, args.length)});
	}
	
	public static void main(String[] args) throws Throwable
	{
		if(args.length < 1)
		{
			printHelpAndExit();
		}
		
		Thread.currentThread().setContextClassLoader(CLASS_LOADER);
		
		switch(args[0])
		{
		case "launch": launch(Arrays.copyOfRange(args, 1, args.length)); return;
		case "main": launchMain(Arrays.copyOfRange(args, 1, args.length)); return;
		
		default:
			System.err.println("Unknown operation: "+args[0]);
			printHelpAndExit();
		}
	}
}
