package eu.javaexperience.electronic.lcd.menu;

import java.util.Map;

import eu.javaexperience.collection.map.SmallMap;
import eu.javaexperience.datareprez.DataCommon;
import eu.javaexperience.datareprez.DataObject;
import eu.javaexperience.electronic.lcd.menu.input.InputEvent;
import eu.javaexperience.interfaces.ExternalDataAttached;
import eu.javaexperience.interfaces.simple.SimpleGet;
import eu.javaexperience.interfaces.simple.publish.SimplePublish1;

public abstract class AbstractLcdMenu<L extends LcdMenu<L>> implements LcdMenu<L>, ExternalDataAttached
{
	protected String id;
	protected LcdMenuHost<L> host;
	protected int index = 0;
	
	protected SimpleGet<L> back;
	
	protected boolean autoBack = true;
	
	protected InputEventNavigationInterpreter nav;
	
	protected int parentIndex = 0;
	
	protected SimplePublish1<AbstractLcdMenu<L>> afterEnter;
	
	public AbstractLcdMenu(String id, InputEventNavigationInterpreter nav)
	{
		this.id = id;
		this.nav = nav;
	}
	
	@Override
	public String getId()
	{
		return id;
	}
	
	@Override
	public void publish(InputEvent a)
	{
		if(null != back && nav.isBack(a))
		{
			enterMenuWithReturn(back.get(), false);
		}
		else if(nav.isUp(a))
		{
			setIndex(++index);
		}
		else if(nav.isDown(a))
		{
			setIndex(--index);
		}
		else if(nav.isSelect(a) && canSelectIndex(index))
		{
			enterMenuWithReturn(getMenuByIndex(index), true);
		}
	}
	
	public abstract boolean canSelectIndex(int index);

	protected void enterMenuWithReturn(LcdMenu menu, boolean trueForward_falseBackward)
	{
		if(null == menu)
		{
			return;
		}
		
		if(menu.canEnter())
		{
			if(trueForward_falseBackward)
			{
				if(menu instanceof LcdSpinnerMenu)
				{
					LcdSpinnerMenu dst = (LcdSpinnerMenu) menu;
					if(dst.isAutoBack())
					{
						dst.setBackMenu(()->this);
					}
				}
			}
			
			if(trueForward_falseBackward && menu instanceof LcdSpinnerMenu)
			{
				((LcdSpinnerMenu)menu).parentIndex = index;
			}
			
			if(null != host)
			{
				host.enterMenu((L) menu);
			}
			
			if(!trueForward_falseBackward && menu instanceof LcdSpinnerMenu)
			{
				((LcdSpinnerMenu)menu).setIndex(parentIndex);
			}
		}
	}

	public abstract LcdMenu getMenuByIndex(int index);

	@Override
	public void init(LcdMenuHost<L> host)
	{
		this.host = host;
		setIndex(0);
		if(null != afterEnter)
		{
			afterEnter.publish(this);
		}
	}
	
	public abstract int filterIndex(int index);
	
	public abstract LcdDisplayState getCurrentDisplayState();
	
	@Override
	public void refresh()
	{
		if(null != host)
		{
			LcdDisplayState state = getCurrentDisplayState();
			if(null != state)
			{
				host.setDisplayState(state);
			}
		}
	}
	
	public LcdMenuHost<L> getMenuHost()
	{
		return host;
	}
	
	public void setIndex(int index)
	{
		this.index = filterIndex(index);
		refresh();
	}
	
	@Override
	public DataObject saveState(DataCommon proto)
	{
		DataObject ret = proto.newObjectInstance();
		ret.putInt("index", index);
		return ret;
	}
	
	@Override
	public void restoreState(DataObject from)
	{
		setIndex(from.optInt("index", 0));
	}

	@Override
	public void dispose(){}

	public void setBackMenu(SimpleGet<L> back)
	{
		this.back = back;
	}

	public int getIndex()
	{
		return index;
	}
	
	public void setAutoBack(boolean autoBack)
	{
		this.autoBack = autoBack;
	}
	
	public AbstractLcdMenu<L> unsetAutoBack()
	{
		this.autoBack = false;
		return this;
	}
	
	public boolean isAutoBack()
	{
		return this.autoBack;
	}
	
	@Override
	public abstract boolean canEnter();

	public void pinBackMenu(SimpleGet<L> back)
	{
		this.autoBack = false;
		this.back = back;
	}
	
	@Override
	public String toString()
	{
		return "AbstractLcdMenu(id:`"+id+"`)";
	}

	protected transient Map<String, Object> extraData;
	
	@Override
	public Map<String, Object> getExtraDataMap()
	{
		if(null == extraData)
		{
			extraData = new SmallMap<>();
		}
		return extraData;
	}
}
