package eu.javaexperience.electronic.lcd.display;

import eu.javaexperience.electronic.lcd.Lcd8Char;
import eu.javaexperience.electronic.lcd.Lcd58Chars.Lcd58AsciiChars;
import eu.javaexperience.electronic.lcd.menu.LcdDisplay;
import eu.javaexperience.electronic.lcd.menu.LcdDisplayState;
import eu.javaexperience.interfaces.simple.getBy.GetBy1;
import eu.javaexperience.log.JavaExperienceLoggingFacility;
import eu.javaexperience.log.LogLevel;
import eu.javaexperience.log.Loggable;
import eu.javaexperience.log.Logger;
import eu.javaexperience.log.LoggingTools;

public class CharacterDisplayState implements LcdDisplayState
{
	protected static Logger LOG = JavaExperienceLoggingFacility.getLogger(new Loggable("LcdDisplayState"));
	
	public String rawDisplayContent;
	
	public Lcd8Char[][] screenContent;
	public int charactersPerLine;
	public int lines;
	public boolean inverted = false;
	
	/**
	 * Assuming display can show all the requested characters
	 * */
	public static int determineRequestedLength(String string, int charactersPerLine)
	{
		int ret = 0;
		for(int i=0;i<string.length();++i)
		{
			char c = string.charAt(i);
			if('\r' == c || '\n' == c)
			{
				if(0 == ret || 0 != ret % charactersPerLine)
				{
					ret = ((ret/charactersPerLine)+1)*charactersPerLine;
				}
			}
			else
			{
				++ret;
			}
		}
		
		return ret;
	}
	
	public static CharacterDisplayState createFromString(GetBy1<Lcd8Char, Character> charmap, String string, int charactersPerLine, boolean inverted)
	{
		return createFromString(charmap, string, charactersPerLine, true, inverted);
	}
	
	public static CharacterDisplayState createFromString(GetBy1<Lcd8Char, Character> charmap, String string, int charactersPerLine, boolean fillSpaces, boolean inverted)
	{
		CharacterDisplayState ret = new CharacterDisplayState();
		int req = determineRequestedLength(string, charactersPerLine);
		int lines = req/charactersPerLine;
		if(lines*charactersPerLine < req)
		{
			++lines;
		}
		
		ret.rawDisplayContent = string;
		ret.lines = lines;
		ret.charactersPerLine = charactersPerLine;
		ret.screenContent = new Lcd8Char[lines][charactersPerLine];
		ret.inverted = inverted;
		
		if(LOG.mayLog(LogLevel.TRACE))
		{
			LoggingTools.tryLogFormat
			(
				LOG,
				LogLevel.TRACE,
				"createFromString(string: `%s`): chars %s, lines: %s, charactersPerLine: %s ",
				string, req, lines, charactersPerLine
			);
		}
		
		int max = lines*charactersPerLine;
		
		int ep = 0;
		for(int i=0;i<string.length();++i)
		{
			if(ep >= max)
			{
				break;
			}
			
			char chr = string.charAt(i);
			if('\r' == chr || '\n' == chr)
			{
				if(0 != ep && 0 == ep % charactersPerLine)
				{
					if(LOG.mayLog(LogLevel.TRACE))
					{
						LoggingTools.tryLogFormat(LOG, LogLevel.TRACE, "new line, but exactly at a line end, skipping");
					}
				}
				else
				{
					int next = ((ep/charactersPerLine)+1)*charactersPerLine;
					if(LOG.mayLog(LogLevel.TRACE))
					{
						LoggingTools.tryLogFormat(LOG, LogLevel.TRACE, "new line, filling space between %s - %s", ep, next);
					}
					if(fillSpaces)
					{
						ret.fillBetween(ep, next, Lcd58AsciiChars.SPACE);
					}
					ep = next;
				}
			}
			else
			{
				Lcd8Char dc = charmap.getBy(chr);
				if(null != dc)
				{
					if(LOG.mayLog(LogLevel.TRACE))
					{
						LoggingTools.tryLogFormat(LOG, LogLevel.TRACE, "placing char `%s` (`%s`) to line %s, to row %s", dc, dc.getChar(), ep/charactersPerLine, ep%charactersPerLine);
					}
					ret.screenContent[ep/charactersPerLine][ep%charactersPerLine] = dc;
					++ep;
				}
			}
		}
		
		return ret;
	}

	public void fillBetween(int from, int toExcluded, Lcd58AsciiChars character)
	{
		for(int ep=from;ep < toExcluded;++ep)
		{
			screenContent[ep/charactersPerLine][ep%charactersPerLine] = character;
		}
	}

	public String getRaw()
	{
		return null == rawDisplayContent?"":rawDisplayContent;
	}

	@Override
	public void render(LcdDisplay display)
	{
		//TODO
		
		
	}

	public static CharacterDisplayState fromData(Lcd8Char[][] content)
	{
		CharacterDisplayState ret = new CharacterDisplayState();
		ret.screenContent = content;
		ret.lines = content.length;
		for(Lcd8Char[] c:content)
		{
			if(null != c)
			{
				if(c.length > ret.charactersPerLine)
				{
					ret.charactersPerLine = c.length;
				}
			}
		}
		return ret;
	}
}
