package eu.javaexperience.classes.dinamic;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import eu.javaexperience.asserts.AssertArgument;
import eu.javaexperience.classes.ClassDescriptor;
import eu.javaexperience.classes.ClassFieldDescriptor;
import eu.javaexperience.classes.ClassSpace;
import eu.javaexperience.collection.CollectionTools;
import eu.javaexperience.collection.enumerations.EnumLike;
import eu.javaexperience.collection.enumerations.SimpleEnumLike;
import eu.javaexperience.collection.map.MapTools;
import eu.javaexperience.datareprez.DataArray;
import eu.javaexperience.datareprez.DataObject;
import eu.javaexperience.interfaces.simple.getBy.GetBy1;
import eu.javaexperience.interfaces.simple.getBy.GetBy2;
import eu.javaexperience.reflect.Mirror;

public class DinamicClassSpace implements ClassSpace
{
	protected final ConcurrentMap<String, ClassDescriptor> knowClasses = new ConcurrentHashMap<>();
	
	protected final GetBy2<ClassDescriptor, DinamicClassSpace, String> classLoader;
	
	public DinamicClassSpace(GetBy1<ClassDescriptor, String> classLoader)
	{
		this.classLoader = (c,e)->classLoader.getBy(e);
	}
	
	public DinamicClassSpace(GetBy2<ClassDescriptor, DinamicClassSpace, String> classLoader)
	{
		this.classLoader = classLoader;
	}
	
	public ClassDescriptor getClassByName(String name)
	{
		return MapTools.getOrCreate(knowClasses, name, (e)->classLoader.getBy(this, e));
	}
	
	public void registerClass(String className, ClassDescriptor classDescriptor)
	{
		classDescriptor.setClassSpace(this);
		knowClasses.putIfAbsent(className, classDescriptor);
	}

	public ClassDescriptor tryGetClass(String key)
	{
		return knowClasses.get(key);
	}

	public ClassDescriptor parseDescriptor(DataObject obj)
	{
		String key = obj.getString("fullName");
		AssertArgument.assertNotNull(key, "fullName");
		ClassDescriptor ret = tryGetClass(key);
		if(null != ret)
		{
			registerClass(key, ret);
			return ret;
		}
		
		ret = parseDescriptorUnit(obj);
		
		registerClass(key, ret);
		
		return ret;
	}
	
	public static GetBy2<ClassDescriptor, DinamicClassSpace, String> createJavaNativeClassLoader(GetBy1<ClassDescriptor, Class> wrapper)
	{
		return new GetBy2<ClassDescriptor, DinamicClassSpace, String>()
		{
			@Override
			public ClassDescriptor getBy(DinamicClassSpace a, String b)
			{
				try
				{
					return wrapper.getBy(Class.forName(b));
				}
				catch(Exception e)
				{
					Mirror.propagateAnyway(e);
					return null;
				}
			}
		};
	}
	
	protected ClassDescriptor parseDescriptorUnit(DataObject obj)
	{
		String key = obj.getString("fullName");
		
		boolean isArray = obj.getBoolean("isArray");
		boolean isEnum = obj.getBoolean("isEnum");

		if(isArray)
		{
			ClassDescriptor cd = fetchOrParseClass(obj.get("arrayComponent"));
			if(null == cd)
			{
				return null;
			}
			DinamicClassDescriptor ret = DinamicClassDescriptor.createArrayType(cd);
			registerClass(key, ret);
			return ret;
		}
		else if(isEnum)
		{
			DataArray raw = obj.getArray("enums");
			EnumLike[] vals = new EnumLike[raw.size()];
			
			for(int i=0;i<vals.length;++i)
			{
				vals[i] = new SimpleEnumLike<>(raw.getObject(i).getString("name"));
			}
			
			DinamicClassDescriptor ret = DinamicClassDescriptor.createEnumeration(key, vals);
			registerClass(key, ret);
			return ret;
		}
		else
		{
			DataArray fields = obj.getArray("fields");
			DinamicClassField[] dcf = new DinamicClassField[fields.size()];
			for(int i=0;i<fields.size();++i)
			{
				DataObject field = fields.getObject(i);
				String dc = field.getString("declaringClass");
				String type = field.getString("type");
				String name = field.getString("name");
				DataArray annotations = field.optArray("annotations");
				dcf[i] = DinamicClassField.create
				(
					field.getInt("modifiers"),
					this,
					type,
					name, 
					dc+"."+name,
					true,
					null == annotations?null:DinamicClassAnnotation.parse(annotations)
				);
			}
			
			ArrayList<ClassDescriptor> supers = new ArrayList<>();
			{
				Object sc = obj.opt("superClass");
				if(null != sc)
				{
					supers.add(fetchOrParseClass(sc));
				}
			}
			
			{
				DataArray arr = obj.getArray("superInterfaces");
				for(int i=0;i<arr.size();++i)
				{
					supers.add(fetchOrParseClass(arr.getObject(i)));
				}
			}
			
			DinamicClassDescriptor ret = DinamicClassDescriptor.createClass
			(
				supers,
				key,
				obj.getInt("modifiers"),
				dcf
			);
			
			for(DinamicClassField f:dcf)
			{
				f.ownerClass = ret;
			}
			
			registerClass(key, ret);
			return ret;
		}
	}
	
	public void discoverAll()
	{
		Set<ClassDescriptor> discoveredClasses = new HashSet<>();
		for(ClassDescriptor c:knowClasses.values())
		{
			discoverType(discoveredClasses, c);
		}
	}
	
	protected void discoverType(Set<ClassDescriptor> discoveredClasses, ClassDescriptor c)
	{
		if(discoveredClasses.contains(c))
		{
			return;
		}
		discoveredClasses.add(c);
		
		c.getSuperTypes();
		c.getComponentType();
		for(ClassFieldDescriptor f:c.getAllField())
		{
			discoverType(discoveredClasses, f.getType());
		}
	}
	
	protected ClassDescriptor fetchOrParseClass(Object o)
	{
		if(o instanceof String)
		{
			return getClassByName((String) o);
		}
		else
		{
			return parseDescriptor((DataObject) o);
		}
	}
	
	public Set<String> listClasses()
	{
		return knowClasses.keySet();
	}

	public List<ClassDescriptor> getLoadedClasses()
	{
		return CollectionTools.inlineAdd(new ArrayList<>(), knowClasses.values());
	}
}
