package eu.javaexperience.classes.java;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import eu.javaexperience.classes.ClassDescriptor;
import eu.javaexperience.classes.ClassDescriptorTools;
import eu.javaexperience.classes.ClassFieldDescriptor;
import eu.javaexperience.classes.ClassSpace;
import eu.javaexperience.collection.enumerations.EnumManager;
import eu.javaexperience.collection.enumerations.EnumTools;
import eu.javaexperience.collection.map.MapTools;
import eu.javaexperience.interfaces.ObjectWithProperty;
import eu.javaexperience.interfaces.simple.getBy.GetBy1;
import eu.javaexperience.interfaces.simple.getBy.GetBy2;
import eu.javaexperience.reflect.FieldSelectTools;
import eu.javaexperience.reflect.Mirror;

public class JavaClassDescriptor implements ClassDescriptor, ObjectWithProperty
{
	protected ClassSpace classSpace;
	protected Class cls;
	protected EnumManager enums;
	protected List<JavaClassField> fields;
	
	protected JavaClassDescriptor(){}
	
	public JavaClassDescriptor(Class cls, List<JavaClassField> fields)
	{
		this.cls = cls;
		this.fields = fields;
	}
	
	@Override
	public String getClassName()
	{
		return cls.getName();
	}

	@Override
	public List<? extends ClassFieldDescriptor> getAllField()
	{
		return fields;
	}

	@Override
	public boolean isEnum()
	{
		return cls.isEnum();
	}

	@Override
	public EnumManager getEnumManager()
	{
		if(null == enums)
		{
			if(null == cls.getEnumConstants())
			{
				enums = null;//EnumTools.EMPTY_ENUM_MANAGER;
			}
			else
			{
				enums = EnumTools.createFromEnumClass(cls);
			}
		}
		return enums;
	}

	@Override
	public boolean isArray()
	{
		return cls.isArray();
	}

	protected ClassDescriptor componentType;
	
	@Override
	public ClassDescriptor getComponentType()
	{
		if(null == componentType)
		{
			if(null == cls.getComponentType())
			{
				return null;
			}
			componentType = getOfJavaClass(cls.getComponentType());
		}
		return componentType;
	}
	
	protected static final ConcurrentMap<Class, JavaClassDescriptor> CLS_DESCRIPTORS = new ConcurrentHashMap<Class, JavaClassDescriptor>();
	
	protected static final GetBy1<JavaClassDescriptor, Class> WRAP_CLASS = new GetBy1<JavaClassDescriptor, Class>()
	{
		@Override
		public JavaClassDescriptor getBy(Class a)
		{
			if(a == null)
			{
				return null;
			}
			JavaClassDescriptor ret = new JavaClassDescriptor();
			ret.cls = a;
			ret.fields = JavaClassField.wrapAll(Mirror.getClassData(a).selectFields(FieldSelectTools.SELECT_ALL_INSTANCE_FIELD));
			return ret;
		}
	};
	public static final GetBy2<ClassDescriptor, ClassSpace, String> JAVA_CLASS_LOADER = new GetBy2<ClassDescriptor, ClassSpace, String>()
	{
		@Override
		public ClassDescriptor getBy(ClassSpace a, String b)
		{
			try
			{
				return WRAP_CLASS.getBy(Class.forName(b));
			}
			catch (ClassNotFoundException e)
			{
				Mirror.propagateAnyway(e);
				return null;
			}
		}
	}; 
	
	public static JavaClassDescriptor getOfJavaClass(Class javaClass)
	{
		return MapTools.getOrCreate(CLS_DESCRIPTORS, javaClass, WRAP_CLASS);
	}

	@Override
	public int getModifiers()
	{
		return cls.getModifiers();
	}

	protected List<ClassDescriptor> supers;
	
	@Override
	public List<ClassDescriptor> getSuperTypes()
	{
		if(null == supers)
		{
			List<ClassDescriptor> add = new ArrayList<>();
			
			Class c = cls.getSuperclass();
			if(null != c)
			{
				add.add(getOfJavaClass(c));
			}
			
			for(Class cl:cls.getInterfaces())
			{
				add.add(getOfJavaClass(cl));
			}
			
			supers = Collections.unmodifiableList(add);
		}
		return supers;
	}

	@Override
	public boolean isInterface()
	{
		return cls.isInterface();
	}

	@Override
	public Object get(String key)
	{
		return ClassDescriptorTools.classPropGet(this, key);
	}

	@Override
	public String[] keys()
	{
		return ClassDescriptorTools.classPropKeys();
	}

	@Override
	public ClassSpace getClassSpace()
	{
		return classSpace;
	}
	
	@Override
	public void setClassSpace(ClassSpace classSpace)
	{
		this.classSpace = classSpace;
	}
}
