package eu.javaexperience.web.server.commons;

import java.io.IOException;
import java.net.ServerSocket;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;

import eu.javaexperience.asserts.AssertArgument;
import eu.javaexperience.collection.map.KeyVal;
import eu.javaexperience.interfaces.simple.SimpleGet;
import eu.javaexperience.interfaces.simple.publish.SimplePublish1;
import eu.javaexperience.io.IOStream;
import eu.javaexperience.io.IOStreamServer;
import eu.javaexperience.io.IOTools;
import eu.javaexperience.io.IpAddressTools;
import eu.javaexperience.io.fd.IOStreamFactory;
import eu.javaexperience.reflect.Mirror;
import eu.javaexperience.resource.pool.ResourcePool;
import eu.javaexperience.resource.pool.SimplifiedResourcePool;
import eu.javaexperience.server.AbstractServer;
import eu.javaexperience.web.server.example.ExampleServlet;
import eu.javaexperience.web.server.http.HttpServingTimeMetric;
import eu.javaexperience.web.server.http.HttpSocketProtocol;
import eu.javaexperience.web.server.http.HttpSocketProtocolHandler;
import eu.javaexperience.web.server.http.WellKnownSocketHttpProtocol;
import eu.javaexperience.web.server.lightningImpl.LightningHttpQueryContext;

/**
 * This is a top level class, contains all the necessary implementaton
 * to start a ServletServer.
 * 
 * A server requires a:
 * 	- SocketImplementation (to listen, accept, close, do io ops with socket)
 * 	- Thread management (how request should be executed)
 * 	- Strategy for request handling
 * */
public class Lightning<T extends IOStream> extends AbstractServer<T>
{
	
	//protected final SturmWaffeProcessingStrategy strategy;
	protected final HttpServlet servlet;
	protected final HttpSocketProtocol protocol;
	
	public Lightning(IOStreamServer<T> srv, int initialWorkerCount, HttpServlet servlet)
	{
		this(srv, WellKnownSocketHttpProtocol.RAW_HTTP, initialWorkerCount, servlet);
	}
	
	public Lightning(IOStreamServer<T> srv, HttpSocketProtocol protocol, int initialWorkerCount, HttpServlet servlet)
	{
		super(srv, initialWorkerCount);
		AssertArgument.assertNotNull(this.protocol = protocol, "protocol");
		AssertArgument.assertNotNull(this.servlet = servlet, "servlet");
	}
	
	
	
	/**
	 * Egy húsos oldal is általában belefér 120 kb-ba, de én fölé lövök, had'
	 * nőjön a termelékenység. (nagyobb oldal és fileok esetén is)
	 * */
	protected static final ResourcePool<byte[]> BUFFER_POOL = new SimplifiedResourcePool<>(new SimpleGet<byte[]>()
	{
		@Override
		public byte[] get()
		{
			return new byte[300*1024];//300 kB
		}
	});
	
	protected ConcurrentMap<Integer, Entry<AtomicInteger,ConcurrentLinkedQueue<T>>> connectionPerIp = 
			new ConcurrentHashMap<>();
	
	/**
	 * Tries to push the connection into a delay queue.
	 * if max connection for the ip is reached the method enqueue
	 * the connection and returns true.
	 * 
	 * If the method returns false the query may proceed immediatly.
	 * 
	 * Other method TODO
	 * 
	 * */
	protected static <T extends IOStream> boolean tryPushConnection
	(
		ConcurrentMap<Integer, Entry<AtomicInteger,ConcurrentLinkedQueue<T>>> connPerIp,
		T socket,
		int ip,
		int maxConnectionPerIp
	)
	{
		Entry<AtomicInteger,ConcurrentLinkedQueue<T>> kv = connPerIp.get(ip);
		if(null == kv)
		{
			Entry<AtomicInteger, ConcurrentLinkedQueue<T>> q =
				new KeyVal<>(new AtomicInteger(), new ConcurrentLinkedQueue<T>());
			
			kv = connPerIp.putIfAbsent(ip, q);
			
			if(null == kv)
			{
				kv = q;
			}
		}
		
		if(isTooManyConnectionOfHost(connPerIp, ip, maxConnectionPerIp))
		{
			kv.getValue().add(socket);
			return true;
		}
		else
		{
			return false;
		}
	}
	
	protected static <T extends IOStream> boolean isTooManyConnectionOfHost
	(
		ConcurrentMap<Integer, Entry<AtomicInteger,ConcurrentLinkedQueue<T>>> connPerIp,
		int ip,
		int maxConnectionPerIp
	)
	{
		return isTooManyConnectionOfHost(connPerIp.get(ip), maxConnectionPerIp);
	}	
	
	protected static <T extends IOStream> boolean isTooManyConnectionOfHost
	(
		Entry<AtomicInteger, ConcurrentLinkedQueue<T>> kv,
		int maxConnectionPerIp
	)
	{
		if(null == kv)
		{
			return false;
		}
		return kv.getKey().incrementAndGet() > maxConnectionPerIp;
	}
	
	protected static <T> T tryPopConnection
	(
		ConcurrentMap<Integer, Entry<AtomicInteger,ConcurrentLinkedQueue<T>>> connPerIp,
		int ip
	)
	{
		Entry<AtomicInteger,ConcurrentLinkedQueue<T>> kv = connPerIp.get(ip);
		if(null != kv)
		{
			kv.getKey().decrementAndGet();
			return kv.getValue().poll();
		}
		
		return null;
	}
	
	protected boolean enableKeepAlive = true;
	
	protected int maxConnectionPerIp = -1;//TODO
	
	public void setKeepAlive(boolean keepAlive)
	{
		this.enableKeepAlive = keepAlive;
	}
	
	/**
	 * disable if 0 or less
	 * */
	public void setMaxConnectionPerIp(int conn)
	{
		if(conn < 0)
		{
			conn = 0;
		}
		this.maxConnectionPerIp = conn;
	}
	
	
	public static void main(String[] args) throws Throwable
	{
		IOStreamServer<IOStream> srv = IOStreamFactory.fromServerSocket(new ServerSocket(8888));
		//SturmWaffeProcessingStrategy strategy = new SwStrategyUnlimited();
		int initialWorkerCount = 5;
		HttpServlet servlet = new ExampleServlet();

		Lightning<IOStream> l = new Lightning<IOStream>(srv, initialWorkerCount, servlet);
		
		l.start();
	}

	protected final SimplePublish1<LightningHttpQueryContext> requestHandler = new SimplePublish1<LightningHttpQueryContext>()
	{
		@Override
		public void publish(LightningHttpQueryContext ssrr)
		{
			try
			{
				servlet.service(ssrr.request, ssrr.response);
			}
			catch(Exception e)
			{
				Mirror.propagateAnyway(e);
			}
		}
	};
	
	@Override
	protected void execute(T subject)
	{
		//TODO accept-encoding: gzip
		
		int ip = 0;
		
		if(maxConnectionPerIp > 0)
		{
			ip = IpAddressTools.parseIPv4AsInt(subject.remoteAddress());
			if(tryPushConnection(connectionPerIp, subject, ip, maxConnectionPerIp))
			{
				System.out.println("pushed:"+IpAddressTools.getIpAddress(ip));
				return;
			}
		}
		
		byte[] buffer;
		try
		{
			buffer = BUFFER_POOL.acquireResource();
		}
		catch (InterruptedException e1)
		{
			e1.printStackTrace();
			return;
		}

		enqueue: while(true)//enqued connection
		{
			LightningHttpQueryContext ssrr = null;
			try
			{

				keepalive:while(true)//keep alive loop
				{
					ssrr = LightningTools.handleRequest(protocol, subject, requestHandler);
					
					ssrr.destroy();
					
					if
					(
							!enableKeepAlive
						||
							"close".equalsIgnoreCase(ssrr.request.getHeader("Connection"))
						|| 
							!ssrr.isLengthGivenAtHeaderSend()
						||
							isTooManyConnectionOfHost(connectionPerIp, ip, maxConnectionPerIp)
					)
					{
						break keepalive;
					}
				}
			}
			catch(LightningConnectionClosed closed)
			{
				//nothing to do
			}
			catch(Throwable e)
			{
				e.printStackTrace();
			}
			finally
			{
				if(null != ssrr)
				{
					IOTools.silentFlush(ssrr);
					IOTools.silentClose(ssrr);
				}
				IOTools.silentFlush(subject);
				IOTools.silentClose(subject);
			}
			
			subject = tryPopConnection(connectionPerIp, ip);
			
			if(null != subject)
			{
				ip = IpAddressTools.parseIPv4AsInt(subject.remoteAddress());
				System.out.println("pop:"+IpAddressTools.getIpAddress(ip));
				continue enqueue;
			}
			else
			{
				break enqueue;
			}
		}
		
		if(null != buffer)
		{
			BUFFER_POOL.releaseResource(buffer);
		}
		
	}
}