package eu.linuxengineering.procfs;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import eu.javaexperience.collection.map.SmallMap;
import eu.javaexperience.io.IOTools;
import eu.javaexperience.parse.ParsePrimitive;
import eu.javaexperience.reflect.Mirror;
import eu.javaexperience.regex.RegexTools;

public class CpuStat
{
	public List<CpuStatEntry> cpuEntries = new ArrayList<>();
	
	public Map<String, Long> values = new SmallMap<>();
	
	public List<CpuStatOtherEntry> otherEntries = new ArrayList<>();
	
	public static class CpuStatOtherEntry
	{
		public String name;
		public List<Long> values = new ArrayList<>();
		
		public static CpuStatOtherEntry parse(String[] comp)
		{
			CpuStatOtherEntry ret = new CpuStatOtherEntry();
			ret.name = comp[0];
			for(int i=1;i<comp.length;++i)
			{
				ret.values.add(ParsePrimitive.tryParseLong(comp[i]));
			}
			
			return ret;
		}
		
		@Override
		public String toString()
		{
			return Mirror.usualToString(this);
		}
	}
	
	public static class CpuStatEntry
	{
		public String name;
		
		public long user;
		
		public long nice;
		public long system;
		public long idle;
		public long iowait;
		public long irq;
		public long softirq;
		
		public long steal;
		public long guest;
		
		protected static List<Field> CPU_FIELDS = new ArrayList<>();
		
		static
		{
			for(Field f:CpuStatEntry.class.getFields())
			{
				if(f.getType() == long.class)
				{
					CPU_FIELDS.add(f);
				}
			}
		}
		
		public static CpuStatEntry parse(String[] comp)
		{
			CpuStatEntry ret = new CpuStatEntry();
			ret.name = comp[0];
			for(int i=1;i<comp.length && i <= CPU_FIELDS.size();++i)
			{
				try
				{
					CPU_FIELDS.get(i-1).set(ret, ParsePrimitive.tryParseLong(comp[i]));
				}
				catch(Exception e)
				{
					e.printStackTrace();
				}
			}
			
			return ret;
		}
		
		@Override
		public String toString()
		{
			return Mirror.usualToString(this);
		}
	}
	
	@Override
	public String toString()
	{
		return Mirror.usualToString(this);
	}
	
	public static CpuStat parseStatData(String data)
	{
		CpuStat ret = new CpuStat();
		
		for(String line:RegexTools.LINUX_NEW_LINE.split(data))
		{
			String[] comp = RegexTools.SPACES.split(line);
			if(0 == comp.length)
			{
				continue;
			}
			
			if(2 == comp.length)
			{
				ret.values.put(comp[0], ParsePrimitive.tryParseLong(comp[1]));
			}
			
			if(comp[0].startsWith("cpu"))
			{
				CpuStatEntry add = CpuStatEntry.parse(comp);
				if(null != add)
				{
					ret.cpuEntries.add(add);
				}
			}
			else
			{
				CpuStatOtherEntry oth = CpuStatOtherEntry.parse(comp);
				if(null != oth)
				{
					ret.otherEntries.add(oth);
				}
			}
		}
		
		return ret;
	}
	
	public static void main(String[] args) throws Throwable
	{
		System.out.println(CpuStat.parseStatData(new String(IOTools.loadFileProcContent("/proc/stat"))));
	}
}
