package eu.linuxengineering.snmp;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;

import eu.javaexperience.interfaces.simple.SimpleGet;
import eu.javaexperience.interfaces.simple.getBy.GetBy1;
import eu.javaexperience.log.JavaExperienceLoggingFacility;
import eu.javaexperience.log.LogLevel;
import eu.javaexperience.log.Loggable;
import eu.javaexperience.log.Logger;
import eu.javaexperience.log.LoggingTools;
import eu.javaexperience.reflect.Mirror;
import eu.javaexperience.reflect.PrimitiveTools;
import eu.javaexperience.reflect.Mirror.BelongTo;
import eu.javaexperience.reflect.Mirror.ClassData;
import eu.javaexperience.reflect.Mirror.MethodSelector;
import eu.javaexperience.reflect.Mirror.Select;
import eu.javaexperience.reflect.Mirror.Visibility;
import eu.javaexperience.text.StringTools;
import eu.linuxengineering.snmp.annotations.SnmpIndex;
import eu.linuxengineering.snmp.annotations.SnmpNodeDetails;
import eu.linuxengineering.snmp.annotations.SnmpSubnode;
import eu.linuxengineering.snmp.nodes.SnmpNodeType;
import eu.linuxengineering.snmp.nodes.SnmpReflectFields;
import net.sf.snmpadaptor4j.api.AttributeAccessor;
import net.sf.snmpadaptor4j.object.SnmpDataType;
import net.sf.snmpadaptor4j.object.SnmpOid;

public class SnmpTools
{
	private SnmpTools(){}
	
	protected static final Logger LOG = JavaExperienceLoggingFacility.getLogger(new Loggable("SnmpTools"));
	
	public static SnmpDispatchNode ensurePath
	(
		SnmpDispatchNode from,
		int[] path
	)
	{
		for(int i=0;i<path.length;++i)
		{
			SnmpNode node = from.subNodes.get(path[i]);
			if(null == node)
			{
				SnmpDispatchNode tmp = new SnmpDispatchNode();
				from.addEntry(path[i], tmp);
				node = tmp;
			}
			
			if(node instanceof SnmpDispatchNode)
			{
				from = (SnmpDispatchNode) node;
			}
			else
			{
				throw new RuntimeException
				(
					"A non SnmpDispatchNode node already present in path `"+Arrays.toString(path)+" at "+i+" "+node
				);
			}
		}
		return from;
	}
	
	/**
	 * Returns the node already presen on the requested path
	 * returns null when nothing present on path and adds the node
	 * return the node found ont the path and node will not be added
	 * */
	public static SnmpNode tryAddToPath
	(
		SnmpDispatchNode from,
		int[] path,
		SnmpNode node
	)
	{
		SnmpDispatchNode to = ensurePath(from, Arrays.copyOf(path, path.length-1));
		Integer tar = path[path.length-1];
		
		SnmpNode ret = null;
		ret = to.subNodes.get(tar);
		if(null != ret && ret != node)
		{
			return ret;
		}
		
		to.addEntry(tar, node);
		
		return null;
	}

	
	public static void addToPath
	(
		SnmpDispatchNode from,
		int[] path,
		SnmpNode node,
		boolean forceOverride
	)
	{
		SnmpDispatchNode to = ensurePath(from, Arrays.copyOf(path, path.length-1));
		Integer tar = path[path.length-1];
		if(to.subNodes.containsKey(tar) && !forceOverride)
		{
			throw new RuntimeException("Node at path "+Arrays.toString(path)+" already exists: "+to.subNodes.get(tar));
		}
		to.addEntry(tar, node);
	}

	public static AttributeAccessor createReadOnlyAttributeAccessor(SnmpOid id, SnmpDataType type, SimpleGet<?> source)
	{
		return new AttributeAccessor()
		{
			@Override
			public void setValue(Object value) throws Exception {}
			
			@Override
			public boolean isWritable()
			{
				return false;
			}
			
			@Override
			public boolean isReadable()
			{
				return true;
			}
			
			@Override
			public Object getValue() throws Exception
			{
				return source.get();
			}
			
			@Override
			public SnmpDataType getSnmpDataType()
			{
				return type;
			}
			
			@Override
			public SnmpOid getOid()
			{
				return id;
			}
			
			@Override
			public Class<?> getJmxDataType()
			{
				return null;
			}
		};
	}

	public static SnmpFinalNode addRoAccessorToPath(SnmpDispatchNode root, int[] id, SnmpDataType type, SimpleGet<Object> getter, boolean forceOverride)
	{
		SnmpOid oid = SnmpOid.newInstance(id);
		SnmpFinalNode ret = new SnmpFinalNode(createReadOnlyAttributeAccessor(oid, type, getter));
		addToPath(root, id, ret, forceOverride);
		return ret;
	}
	
	public static SnmpFinalNode createFinalNode(SnmpOid oid, SnmpDataType type, SimpleGet<?> getter)
	{
		return new SnmpFinalNode(createReadOnlyAttributeAccessor(oid, type, getter));
	}
	
	public static SnmpNode addSnmpObjectToPath(SnmpMibDispatch root, int[] path, Object node, boolean forceOverride)
	{
		SnmpNode add = wrapSnmpObject(node); 
		addToPath(root, path, add, forceOverride);
		return add;
	}
	
	public static final MethodSelector SELECT_ALL_PUBLIC_INSTANCE_METHOD = new MethodSelector
	(
		true,
		Visibility.Public,
		BelongTo.Instance,
		Select.IsNot,
		Select.All,
		Select.All,
		Select.All,
		Select.All
	);
	
	public static SnmpDataType recogniseSnmpDataType(Class cls)
	{
		cls = PrimitiveTools.toObjectClassType(cls, cls);
		
		if(String.class == cls)
		{
			return SnmpDataType.octetString;
		}
		if
		(
				Boolean.class == cls
			||
				Byte.class == cls
			||
				Character.class == cls
			||
				Short.class == cls
			||
				Integer.class == cls
		)
		{
			return SnmpDataType.integer32;
		}
		
		if
		(
			Long.class == cls
		)
		{
			return SnmpDataType.counter64;
		}
		
		if(SnmpOid.class == cls)
		{
			return SnmpDataType.objectIdentifier;
		}
		
		return null;
	}
	
	protected static SnmpNode createReflectNode
	(
		String name,
		String description,
		SnmpNodeType type,
		SimpleGet<Integer> itemsCount
	)
	{
		SnmpRequestSensitiveDispatchCollection ret = new SnmpRequestSensitiveDispatchCollection();
		if(!StringTools.isNullOrTrimEmpty(name))
		{
			ret.addNodeFactory(1, (oid)->createFinalNode(oid, SnmpDataType.octetString, ()->name));
		}
		
		if(!StringTools.isNullOrTrimEmpty(description))
		{
			ret.addNodeFactory(2, (oid)->createFinalNode(oid, SnmpDataType.octetString, ()->description));
		}
		
		ret.addNodeFactory(3, (oid)->createFinalNode(oid, SnmpDataType.octetString, ()->type.name()));
		ret.addNodeFactory(4, (oid)->createFinalNode(oid, SnmpDataType.integer32, itemsCount));
		return ret;
	}
	
	/*protected static SnmpNode wrapConstant
	(
		SimpleGet<Object> source,
		String name,
		String description
	)
	{
		return new SnmpNode()
		{
			@Override
			public boolean hasSubNodes()
			{
				return false;
			}
			
			@Override
			public Entry<Integer, SnmpNode> getSubNodeGte(SnmpPathDispatch index)
			{
				return null;
			}
			
			@Override
			public AttributeAccessor getAccessor()
			{
				return null;
			}
		};
	}*/
	
	/**
	 * Well, looks like there's a bug or somethin' in the SNMP agent server
	 * because can't accept SnmpOid object marked with the type of
	 * {@link SnmpDataType#objectIdentifier} even when providing with strings.
	 * So now i propagate them as an octetString 'til this situation gonna be
	 * resolved. 
	 * */
	public static GetBy1<SnmpNode, SnmpOid> wrapWithSnmpObjectCreator(Object o)
	{
		if(o instanceof SnmpRelativeOid)
		{
			return (oid)->createFinalNode(oid, SnmpDataType.octetString, ()->((SnmpRelativeOid)o).resolve(oid).toString());
		}
		
		return (oid)->wrapSnmpObject(o);
	}
	
	protected static SnmpNode wrapCollectionReturning
	(
		Object node,
		Method m
	)
	{
		return new SnmpProxyNode()
		{
			protected Collection prev;
			
			public synchronized void beforeAccess()
			{
				try
				{
					Collection now = (Collection) m.invoke(node);
				
					if(null == original || !Mirror.equals(prev, now))
					{
						SnmpRequestSensitiveDispatchCollection ret = new SnmpRequestSensitiveDispatchCollection();
						SnmpNodeDetails info = m.getAnnotation(SnmpNodeDetails.class);
						ret.addNodeFactory
						(
							0,
							createReflectNode
							(
								null == info?null:info.name(),
								null == info?null:info.description(),
								SnmpNodeType.ENUMERATION,
								()->now.size()
							)
						);
						
						int i = 0;
						for(Object o:now)
						{
							if(null != o)
							{
								ret.addNodeFactory(++i, wrapWithSnmpObjectCreator(o));
							}
						}
						
						original = ret;
					}
				}
				catch(Exception e)
				{
					Mirror.propagateAnyway(e);
				}
			}
			
			@Override
			public String toString()
			{
				return "SnmpTools.wrapCollectionReturning(node: `"+node+"`, method: `"+m+"`)";
			}
		};
	}
	
	/**
	 * 
	 * nocovered cases:
	 * 	- constant values: int, boolean, double, String (enum as string) 
	 * 	- collection or array
	 * 
	 * covered:
	 * 	- object contains methods returns int, string, collection values 
	 * 
	 * */
	public static SnmpNode wrapSnmpObject(Object node)
	{
		if(node instanceof SnmpNode)
		{
			return (SnmpNode) node;
		}
		
		/*if(node instanceof Collection)
		{
			return wrapCollection((Collection) node);
		}*/
		
		return wrapSnmpBeam(node);
	}
	
	public static final GetBy1<SnmpNode, Object> WRAP_OBJECT_TO_SNMP = (obj) -> wrapSnmpObject(obj);
	
	public static GetBy1<SnmpNode, SnmpOid> createRelativeDemandProxy
	(
		GetBy1<SnmpNode, Object> nodeCreator,
		SimpleGet<Object> source,
		boolean useCache
	)
	{
		return (f)->new SnmpProxyNode()
		{
			protected Object prev;
			
			public void beforeAccess()
			{
				Object now = null;
				try
				{
					now = source.get();
				}
				catch(Exception e)
				{
					Mirror.propagateAnyway(e);
				}
				
				if(null == original || !useCache || !Mirror.equals(prev, now))
				{
					original = nodeCreator.getBy(now);
					prev = now;
				}
			}
			
			@Override
			public String toString()
			{
				return "createRelativeDemandProxy(nodeCreator: `"+nodeCreator+"`, source: `"+source+"`, useCache: `"+useCache+"`)";
			}
		};
	}
	
	/**
	 * Intentionally for internal usage, but you can use for other SNMP related
	 * functions, but don't abuse and use in other facility type, that can
	 * who reads the logs, because errors written to the "SnmpTools" label. 
	 * */
	@Deprecated
	public static SimpleGet<Object> wrapMethodGetterWithLogging(Method m, Object subject)
	{
		return ()->
		{
			try
			{
				return m.invoke(subject);
			}
			catch(Exception e)
			{
				LoggingTools.tryLogFormatException
				(
					LOG,
					LogLevel.ERROR,
					e,
					"Exception while invoking method `%s` on `%s`",
					m,
					subject
				);
				Mirror.propagateAnyway(e);
				return null;
			}
		};
	}
	
	public static SnmpNode wrapSnmpBeam(Object node)
	{
		ClassData cd = Mirror.getClassData(node.getClass());
		
		Method[] methods = cd.select(SELECT_ALL_PUBLIC_INSTANCE_METHOD);
		
		SnmpRequestSensitiveDispatchCollection ret = new SnmpRequestSensitiveDispatchCollection();
		
		int[] itemsCount = new int[]{0};
		
		if(node instanceof SnmpReflectFields)
		{
			SnmpReflectFields details = (SnmpReflectFields) node;
			SnmpNode cre = createReflectNode
			(
				details.getName(),
				details.getDescription(),
				SnmpNodeType.COLLECTION,
				()->itemsCount[0]
			);
			
			ret.addNodeFactory(0, cre);
		}
		
		for(Method m:methods)
		{
			if(0 != m.getParameters().length)
			{
				continue;
			}
			
			SnmpIndex index = m.getAnnotation(SnmpIndex.class);
			if(null != index)
			{
				//TODO if subnode, extract, otherwise just create a FinalNode
				//that gets the value
				
				Class retType = m.getReturnType();
				
				SnmpDataType type = recogniseSnmpDataType(retType);
				
				SnmpNodeType nodeType = SnmpNodeType.EXACT_VALUE;
				SimpleGet<Integer> nodeItemCount = ()->1;
				GetBy1<SnmpNode, SnmpOid> getter = null;
				
				if(Collection.class.isAssignableFrom(retType))
				{
					ret.addNodeFactory(index.index(), wrapCollectionReturning(node, m));
					++itemsCount[0];
					continue;
				}
				else if(SnmpRelativeOid.class.isAssignableFrom(retType))
				{
					getter = (oid) ->
					createRelativeDemandProxy
					(
						(o)->wrapWithSnmpObjectCreator(o).getBy(oid),
						wrapMethodGetterWithLogging(m, node),
						true
					).getBy(oid);
				}
				else if(null == type)
				{
					getter = createRelativeDemandProxy(WRAP_OBJECT_TO_SNMP, wrapMethodGetterWithLogging(m, node), true);
				}
				else
				{
					getter = (oid)->SnmpFinalNode.wrap(oid, type, wrapMethodGetterWithLogging(m, node));
				}
				
				if(null == getter)
				{
					continue;
				}
								
				++itemsCount[0];
				
				boolean subnode = null != m.getAnnotation(SnmpSubnode.class);
				
				if(subnode)
				{
					SnmpRequestSensitiveDispatchCollection add = new SnmpRequestSensitiveDispatchCollection();
					SnmpNodeDetails info = m.getAnnotation(SnmpNodeDetails.class);
					if(null != info)
					{
						add.addNodeFactory
						(
							0,
							createReflectNode
							(
								info.name(),
								info.description(),
								nodeType,
								nodeItemCount
							)
						);
						
					}
					
					add.addNodeFactory(1, getter);
					
					ret.addNodeFactory(index.index(), add);
				}
				else
				{
					ret.addNodeFactory(index.index(), getter);
				}
			}
		}
		
		return ret;
	}
}
