package eu.linuxengineering.snmp.nodes.linux;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import eu.javaexperience.cache.CacheOverTime;
import eu.javaexperience.io.IOTools;
import eu.javaexperience.reflect.Mirror;
import eu.linuxengineering.procfs.CpuStat;
import eu.linuxengineering.procfs.CpuStat.CpuStatEntry;
import eu.linuxengineering.snmp.annotations.SnmpIndex;
import eu.linuxengineering.snmp.annotations.SnmpNodeDetails;
import eu.linuxengineering.snmp.annotations.SnmpSubnode;
import eu.linuxengineering.snmp.nodes.SnmpReflectFields;

public class CpuStatNode implements SnmpReflectFields
{
	protected CacheOverTime<File, CpuStatAttachment> cache;
	
	protected static class CpuStatAttachment
	{
		protected CpuStat stat;
		
		protected List<CpuSnmpInfo> cpuInfos = new ArrayList<>();
		
		public static CpuStatAttachment parseStatData(File f)
		{
			CpuStatAttachment attach = new CpuStatAttachment();
			try
			{
				attach.stat = CpuStat.parseStatData(new String(IOTools.loadFileProcContent(f.toString())));
			}
			catch (IOException e)
			{
				Mirror.propagateAnyway(e);
			}
			
			attach.cpuInfos = CpuSnmpInfo.wrapAll(attach.stat.cpuEntries);
			
			return attach;
		}
		
	}
	
	public CpuStatNode(String proc_stat_file, int maxAge)
	{
		cache = new CacheOverTime<File, CpuStatAttachment>
		(
			new File(proc_stat_file),
			maxAge,
			(f)->f.lastModified(),
			(f)->CpuStatAttachment.parseStatData(f)
		);
	}
	
	@Override
	public String getName()
	{
		return "Processor usage informations";
	}
	
	@Override
	public String getDescription()
	{
		return "Informations about the processors user, system, nice, steal, etc times per processor and overall processing informations";
	}	
	
	public static class CpuSnmpInfo implements SnmpReflectFields
	{
		public CpuStatEntry cpu;
		
		public CpuSnmpInfo(CpuStatEntry stat)
		{
			this.cpu = stat;
		}
		
		public static List<CpuSnmpInfo> wrapAll(List<CpuStatEntry> cpuEntries)
		{
			List<CpuSnmpInfo> ret = new ArrayList<>();
			for(CpuStatEntry ent:cpuEntries)
			{
				ret.add(new CpuSnmpInfo(ent));
			}
			
			return ret ;
		}

		@Override
		public String getName()
		{
			return cpu.name;
		}

		@Override
		public String getDescription()
		{
			return "Detailed CPU usage of `"+cpu.name+"`";
		}
		
		@SnmpSubnode
		@SnmpIndex(index=1)
		@SnmpNodeDetails(name="user")
		public long getUserUsage()
		{
			return cpu.user;
		}
		
		@SnmpSubnode
		@SnmpIndex(index=2)
		@SnmpNodeDetails(name="nice")
		public long getNiceUsage()
		{
			return cpu.nice;
		}
		
		@SnmpSubnode
		@SnmpIndex(index=3)
		@SnmpNodeDetails(name="system")
		public long getSystemUsage()
		{
			return cpu.system;
		}
		
		@SnmpSubnode
		@SnmpIndex(index=4)
		@SnmpNodeDetails(name="idle")
		public long getIdleUsage()
		{
			return cpu.idle;
		}

		@SnmpSubnode
		@SnmpIndex(index=5)
		@SnmpNodeDetails(name="iowait")
		public long getIowaitUsage()
		{
			return cpu.iowait;
		}

		@SnmpSubnode
		@SnmpIndex(index=6)
		@SnmpNodeDetails(name="irq")
		public long getIrqUsage()
		{
			return cpu.irq;
		}
		
		@SnmpSubnode
		@SnmpIndex(index=7)
		@SnmpNodeDetails(name="softIrq")
		public long getSoftIrqUsage()
		{
			return cpu.softirq;
		}
		
		@SnmpSubnode
		@SnmpIndex(index=8)
		@SnmpNodeDetails(name="steal")
		public long getStealUsage()
		{
			return cpu.steal;
		}
		
		@SnmpSubnode
		@SnmpIndex(index=9)
		@SnmpNodeDetails(name="guest")
		public long getGuestUsage()
		{
			return cpu.guest;
		}
	}
	
	@SnmpSubnode
	@SnmpIndex(index = 1)
	@SnmpNodeDetails(name="Per cpu usage information")
	public List<CpuSnmpInfo> getPerCpuInfos()
	{
		return cache.get().cpuInfos;
	}
	
	@SnmpSubnode
	@SnmpIndex(index = 2)
	@SnmpNodeDetails(name="Currently running processes")
	public int getRunningProcesses()
	{
		return ((Number)cache.get().stat.values.get("procs_running")).intValue();
	}
	
	@SnmpSubnode
	@SnmpIndex(index = 3)
	@SnmpNodeDetails(name="Currently blocked processes")
	public int getBlockedProcesses()
	{
		return ((Number)cache.get().stat.values.get("procs_blocked")).intValue();
	}
	
	@SnmpSubnode
	@SnmpIndex(index = 4)
	@SnmpNodeDetails
	(
		name="Overall created processes",
		description="Numbers of process entities (processes and threads) since the system started"
	)
	public long getOverallCreatedProcesses()
	{
		return cache.get().stat.values.get("processes");
	}
	
	@SnmpSubnode
	@SnmpIndex(index = 5)
	@SnmpNodeDetails
	(
		name="Overall processes context switches",
		description="Number of process entity context switches since the system started"
	)
	public long getOverallContextSwitches()
	{
		return cache.get().stat.values.get("ctxt");
	}
	
	@SnmpSubnode
	@SnmpIndex(index = 6)
	@SnmpNodeDetails
	(
		name="Boot time",
		description="Boot time in secounds since the UNIX epoch"
	)
	public long getBootTime()
	{
		return cache.get().stat.values.get("btime");
	}
}
