package eu.javaexperience.webgsdb;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;

import org.json.JSONArray;

import eu.javaexperience.arrays.ArrayTools;
import eu.javaexperience.collection.enumerations.EnumTools;
import eu.javaexperience.collection.map.OneShotMap;
import eu.javaexperience.collection.map.SmallMap;
import eu.javaexperience.datareprez.DataArray;
import eu.javaexperience.datareprez.DataCommon;
import eu.javaexperience.datareprez.DataObject;
import eu.javaexperience.datareprez.convertFrom.DataReprezType;
import eu.javaexperience.datareprez.convertFrom.ObjectLike;
import eu.javaexperience.datareprez.convertFrom.ReadWriteObjectLike;
import eu.javaexperience.datareprez.jsonImpl.DataArrayJsonImpl;
import eu.javaexperience.interfaces.ExternalDataAttached;
import eu.javaexperience.log.LogLevel;
import eu.javaexperience.reflect.CastTo;
import eu.javaexperience.reflect.Caster;
import eu.javaexperience.reflect.Mirror;
import eu.javaexperience.semantic.references.MayNotModified;
import eu.javaexperience.verify.TranslationFriendlyValidationEntry;
import eu.javaexperience.verify.Validatable;
import hu.ddsi.java.database.FieldData;
import hu.ddsi.java.database.GenericStorable;
import hu.ddsi.java.database.GenericStorage;
import hu.ddsi.java.database.GenericStorageMappingData;
import hu.ddsi.java.database.GenericStoreDatabase;
import hu.ddsi.java.database.GenericStoreException;
import hu.ddsi.java.database.GsdbModel;
import hu.ddsi.java.database.fieldAnnotations.GenericStoreIgnore;
import eu.javaexperience.verify.LanguageTranslatableValidationEntry;
import eu.jvx.js.lib.TeaVmTools;
import eu.jvx.js.lib.reflect.TeaVmMockedReflection;

public abstract class WebDbModel extends GsdbModel implements ReadWriteObjectLike, Cloneable, Validatable<LanguageTranslatableValidationEntry>, ExternalDataAttached
{
	@MayNotModified
	public Date createdOn = new Date();
	
	@MayNotModified
	public Date lastModify = new Date();
	
	@Override
	public void beforeStored(GenericStoreDatabase db)
	{
		super.beforeStored(db);
		lastModify = new Date();
	}
	
	private static final long serialVersionUID = 1L;
	
	protected static class GsdbModelMappingData
	{
		protected HashMap<String, Field> getFieldValue = new HashMap<>(); 
		public String[] keys;
	}
	
	protected static final ConcurrentHashMap<Class<? extends GenericStorable>, GsdbModelMappingData> mappingData = new ConcurrentHashMap<>();
	
	protected static GsdbModelMappingData getMappingData(Class<? extends GenericStorable> cls) throws InstantiationException, IllegalAccessException
	{
		GsdbModelMappingData ret = mappingData.get(cls);
		
		if(null == ret)
		{
			ret = new GsdbModelMappingData();
			FieldData[] d = GenericStorageMappingData.getOrCollectAllClassData(cls).allFd;
			ret.keys = new String[d.length+2];
			
			ret.keys[0] = "do";
			ret.keys[1] = "class";
			int n = 1;
			for(int i=0;i<d.length;++i)
			{
				Field f = d[i].f;
				if(RemoteGsdbTools.isWebgsdbUseField(f))
				{
					ret.keys[++n] = f.getName();
					ret.getFieldValue.put(f.getName(), f);
				}
			}
			
			if(n != ret.keys.length)
			{
				ret.keys = Arrays.copyOfRange(ret.keys, 0, n+1);
			}
		}
		return ret;
	}
	
	public static <G extends WebDbModel> G parse(GenericStoreDatabase gdb, Class<G> cls, DataObject from) throws InstantiationException, IllegalAccessException, GenericStoreException
	{
		G ret = cls.newInstance();
		
		GsdbModelMappingData mapping = getMappingData(cls);
		for(Entry<String, Field> kv:mapping.getFieldValue.entrySet())
		{
			String key = kv.getKey();
			ret.set(key, from.get(key));
		}
		
		return ret;
	}
	
	public Long getModelId()
	{
		return GenericStorage.getID(this);
	}
	
	@Override
	public Object get(String key)
	{
		if("do".equals(key))
		{
			return GenericStorage.getID(this);
		}
		
		if("class".equals(key))
		{
			return this.getClass().getName();
		}
		
		GsdbModelMappingData mapping = null;
		
		try
		{
			mapping = getMappingData(getClass());
		}
		catch(Exception e)
		{
			Mirror.propagateAnyway(e);
		}
		
		Field f = mapping.getFieldValue.get(key);
		
		if(null == f)
		{
			return getExtraDataMap().get(key);
		}
		
		try
		{
			Object ret = null;
			if(TeaVmTools.IS_FRONTEND)
			{
				return TeaVmMockedReflection.getObjectField(this, key);
			}
			else
			{
				ret = f.get(this);
			}
			
			if(null != ret)
			{
				if(ret.getClass().isEnum())
				{
					return ((Enum)ret).name();
				}
				else if(ret.getClass().isArray())
				{
					Class<?> comp = ret.getClass().getComponentType();
					if(GenericStorable.class.isAssignableFrom(comp))
					{
						Long[] r = new Long[((Object[])ret).length];
						for(int i=0;i<r.length;++i)
						{
							GenericStorable elem = ((GenericStorable[])ret)[i];
							if(null == elem)
							{
								r[i] = -1l;
							}
							else
							{
								r[i] = GenericStorage.getID(elem);
							}
						}
						return r;
					}
					
					return ret;
				}
				if(ret instanceof GenericStorable)
				{
					return GenericStorage.getID(((GenericStorable)ret));
				}
			}
			return ret;
		}
		catch (Exception e)
		{
			e.printStackTrace();
			return null;
		}
	}

	@Override
	public DataReprezType getDataReprezType()
	{
		return DataReprezType.OBJECT;
	}

	@Override
	public boolean has(String key)
	{
		return ArrayTools.contains(keys(), key);
	}

	@Override
	public String[] keys()
	{
		GsdbModelMappingData mapping = null;
		try
		{
			mapping = getMappingData(getClass());
		}
		catch (Exception e)
		{
			Mirror.propagateAnyway(e);
		}
		return mapping.keys;
	}

	@Override
	public int size()
	{
		return keys().length;
	}

	@Override
	public boolean set(String key, Object src)
	{
		return set(key, src, false);
	}
	
	public boolean set(String key, Object src, boolean filter)
	{
		try
		{
			GsdbModelMappingData mapping = getMappingData(this.getClass());
			Field f = mapping.getFieldValue.get(key);
			
			if(null != f)
			{
				if(filter && null != f.getAnnotation(MayNotModified.class))
				{
					return false;
				}
				
				if(null != src)
				{
					src = tryCast(this, f.getType(), src);

					if(TeaVmTools.IS_FRONTEND)
					{
						return TeaVmMockedReflection.setObjectField(this, key, src);
					}
					else
					{
						f.set(this, src);
						return true;
					}
				}
			}
			else
			{
				getExtraDataMap().put(key, src);
			}
		}
		catch(Exception e)
		{
			//Mirror.propagateAnyway(e);
		}
		return false;
	}
	
	protected static <T> Object tryCast(WebDbModel subject, Class reqType, Object src) throws Exception
	{
		if(GenericStorable.class.isAssignableFrom(reqType))
		{
			Long id = (Long) CastTo.Long.cast(src);
			if(null != id)
			{
				GenericStoreDatabase gdb = GenericStorage.getOwnerDatabase(subject);
				if(null != gdb)
				{
					return GenericStorage.getObjectByIDDescendantOf(id, reqType, gdb);
				}
			}
		}
		else
		{
			if(reqType.isEnum())
			{
				return EnumTools.recogniseSymbol(reqType, src);
			}
			else
			{
				if(String.class == reqType && src instanceof DataCommon)
				{
					byte[] ret = ((DataCommon)src).toBlob();
					return null  == ret?null:new String(ret);
				}
				
				Caster c = CastTo.getCasterRestrictlyForTargetClass(reqType);
				if(null != c)
				{
					return c.cast(src);
				}
			}
		}
		
		if(reqType.isEnum())
		{
			if(src instanceof DataObject)
			{
				return EnumTools.recogniseSymbol(reqType, ((DataObject) src).optString("name"));
			}
		}
		else if(reqType.isArray())
		{
			DataArray arr = null;
			if(src instanceof String)
			{
				arr = new DataArrayJsonImpl(new JSONArray(src.toString()));
			}
			else if(src instanceof DataArray)
			{
				arr = (DataArray) src;
			}
			
			Class nType = reqType.getComponentType();
			
			T[] ret = (T[]) Array.newInstance(nType, arr.size());
			for(int i=0;i<ret.length;++i)
			{
				ret[i] = (T) tryCast(subject, nType, arr.get(i));
			}
			
			return ret;
		}
		else if(null != reqType && null != src && reqType.isAssignableFrom(src.getClass()))
		{
			return src;
		}
		
		return null;
	}

	@Override
	public void unset(String key)
	{
		set(key, null);
	}
	
	public WebDbModel clone()
	{
		try
		{
			return (WebDbModel) super.clone();
		}
		catch (CloneNotSupportedException e)
		{
			Mirror.propagateAnyway(e);
			return null;
		}
	}
	
	public void loadFrom(ObjectLike from)
	{
		for(String k:from.keys())
		{
			set(k, from.get(k));
		}
	}
	
	public void loadFromUser(ObjectLike from)
	{
		for(String k:from.keys())
		{
			set(k, from.get(k), true);
		}
	}
	
	@GenericStoreIgnore
	protected transient Map<String, Object> extraData;
	
	@Override
	public Map<String, Object> getExtraDataMap()
	{
		if(null == extraData)
		{
			extraData = new SmallMap<>();
		}
		return extraData;
	}
	
	//TODO taglink figylmen kívül hagyása: töröljük azokat is.
	public static boolean letDeleteIfNoRef(GenericStorable gs, Collection<LanguageTranslatableValidationEntry> validate)
	{
		try
		{
			List<GenericStorable> gss = new ArrayList<>();
			GenericStorage.getReferences(gs, gss);
			if(gss.isEmpty())
			{
				GenericStoreDatabase gdb = GenericStorage.getOwnerDatabase(gs);
				GenericStorage.removeObject(gs, gdb);
				return true;
			}
			
			for(GenericStorable ref:gss)
			{
				validate.add
				(
					new LanguageTranslatableValidationEntry
					(
						LogLevel.ERROR,
						new TranslationFriendlyValidationEntry
						(
							"reference",
							"db_referenced_in_use",
							new OneShotMap<String, String>("str", ref.toString())
						)
					)
				);
			}
			
			return false;
		}
		catch(Exception e)
		{
			Mirror.propagateAnyway(e);
			return false;
		}
	}
	
	public abstract boolean delete(Collection<LanguageTranslatableValidationEntry> validate);

	public String getModelClass()
	{
		return (String) get("class");
	}
}
