package eu.javaexperience.webgsdb;

import hu.ddsi.java.database.FieldData;
import hu.ddsi.java.database.GenericStorage;
import hu.ddsi.java.database.GenericStorageMappingData;
import hu.ddsi.java.database.GenericStoreDatabase;
import eu.javaexperience.query.LogicalGroup;
import hu.ddsi.java.database.WrappedClassData;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import eu.javaexperience.classes.ClassDescriptor;
import eu.javaexperience.collection.CollectionTools;
import eu.javaexperience.collection.map.ConcurrentMapTools;
import eu.javaexperience.datareprez.DataArray;
import eu.javaexperience.datareprez.DataObject;
import eu.javaexperience.generic.annotations.Ignore;
import eu.javaexperience.interfaces.simple.SimpleGet;
import eu.javaexperience.interfaces.simple.getBy.GetBy1;
import eu.javaexperience.reflect.CastTo;
import eu.javaexperience.reflect.FieldSelectTools;
import eu.javaexperience.reflect.Mirror;
import eu.javaexperience.verify.TranslationFriendlyValidationEntry;
import eu.javaexperience.verify.ValidationResult;
import eu.javaexperience.webgsdb.annotations.FrontendStructureType;
import eu.javaexperience.webgsdb.frontend.modellayer.dinamic.ClassSpace;
import eu.javaexperience.webgsdb.frontend.modellayer.dinamic.ClassSpaceTools;
import eu.javaexperience.webgsdb.frontend.modellayer.java.JavaClassDescriptor;
import eu.javaexperience.webgsdb.frontend.modellayer.java.JavaClassField;

public class GenericModelManagementApi implements ModelManagementApi
{
	protected SimpleGet<GenericStoreDatabase> srcGdb;
	protected Map<String, Class<? extends WebDbModel>> fetch = new HashMap<>();
	protected List<String> managedClasses = new ArrayList<>();
	protected SimpleGet<Set<String>> permissions;
	
	public GenericModelManagementApi(SimpleGet<GenericStoreDatabase> srcGdb, SimpleGet<Set<String>> permissions, Class<? extends WebDbModel>... classes)
	{
		this.srcGdb = srcGdb;
		this.permissions = permissions;
		for(Class<? extends WebDbModel> c:classes)
		{
			fetch.put(c.getSimpleName(), c);
			fetch.put(c.getName(), c);
			managedClasses.add(c.getName());
		}
	}
	
	public GenericStoreDatabase getDatabase()
	{
		return srcGdb.get();
	}
	
	@Override
	public String[] getManagedModels()
	{
		return managedClasses.toArray(Mirror.emptyStringArray);
	}
	
	@Deprecated
	public static class ModelMetadata
	{
		public WrappedClassData modelClass;
		public FieldData[] fields;
		
		protected static GetBy1<Boolean, Field> FIELD_SELECTOR = new GetBy1<Boolean, Field>()
		{
			@Override
			public Boolean getBy(Field a)
			{
				int mod = a.getModifiers();
				if(!Modifier.isPublic(mod))
				{
					return false;
				}
				
				if(Modifier.isStatic(mod))
				{
					return false;
				}
				return true;
			}
		}; 
		
		protected static ConcurrentMap<Class, ModelMetadata> CACHE = new ConcurrentHashMap<>();
		
		public static ModelMetadata collect(Class c) throws InstantiationException, IllegalAccessException
		{
			return ConcurrentMapTools.getOrCreate
			(
				CACHE,
				c,
				(cls) ->
				{
					ModelMetadata ret = new ModelMetadata();
					ret.modelClass = WrappedClassData.wrap(cls, FIELD_SELECTOR);
					ret.fields = GenericStorageMappingData.getOrCollectAllClassData(cls).storedFd;
					return ret;
				}
			);
		}
	}
	
	@Deprecated
	@Override
	public ModelMetadata getModelMetadata(String cls) throws Exception
	{
		//TODO wrap classData
		return ModelMetadata.collect(fetchClass(cls));
	}
	
	protected List<ClassDescriptor> accessibleClasses;
	
	//TODO replace with a ClassSpace with a custom loader witch wraps only the public fields of the model
	protected ClassSpace classSpace = ClassSpaceTools.createUsualClassSpace
	(
		ClassSpace.createJavaNativeClassLoader
		(
			new GetBy1<ClassDescriptor, Class>()
			{
				@Override
				public ClassDescriptor getBy(Class cls)
				{
					if(cls == null)
					{
						return null;
					}
					
					List<Field> fs = new ArrayList<>();
					for(Field f:Mirror.getClassData(cls).selectFields(FieldSelectTools.SELECT_ALL_INSTANCE_FIELD))
					{
						if(Modifier.isPublic(f.getModifiers()))
						{
							fs.add(f);
						}
					}
					
					return new JavaClassDescriptor(cls, JavaClassField.wrapAll(fs.toArray(Mirror.emptyFieldArray)));
				}
			}
		)
	);
	
	protected void discoverAndWrapShallow
	(
		Map<Class, ClassDescriptor> map,
		Class c
	)
	{
		if(null == c)
		{
			return;
		}
		
		if(!map.containsKey(c))
		{
			map.put(c, classSpace.getClassByName(c.getName()));
			
			discoverAndWrapShallow(map, c.getComponentType());
			
			discoverAndWrapShallow(map, c.getSuperclass());
			for(Class i:c.getInterfaces())
			{
				discoverAndWrapShallow(map, i);
			}
			
			for(Field f:Mirror.getClassData(c).select(FieldSelectTools.SELECT_ALL_INSTANCE_FIELD))
			{
				if(Modifier.isPublic(f.getModifiers()))
				{
					discoverAndWrapShallow(map, f.getType());
					//XXX here we can ad more class to discover
					FrontendStructureType ann = f.getAnnotation(FrontendStructureType.class);
					if(null != ann)
					{
						discoverAndWrapShallow(map, ann.structClass());
					}
				}
			}
		}
	}
	
	@Override
	public List<ClassDescriptor> getAllAccessableClass()
	{
		if(null == accessibleClasses)
		{
			Map<Class, ClassDescriptor> map = new HashMap<>();
			for(Class c: fetch.values())
			{
				discoverAndWrapShallow(map, c);
			}
			
			accessibleClasses = Collections.unmodifiableList(CollectionTools.inlineAdd(new ArrayList<>(), map.values()));
		}
		return accessibleClasses;
	}
	
	protected Class<? extends WebDbModel> fetchClass(String name)
	{
		Class<? extends WebDbModel> ret = fetch.get(name);
		if(null == ret)
		{
			throw new RuntimeException("Unknown model class: "+name);
		}
		return ret;
	}
	
	protected long fetchId(DataObject obj)
	{
		Object o = obj.get("do");
		o = CastTo.Long.cast(o);
		if(null == o)
		{
			throw new RuntimeException("id not specified.");
		}
		
		return (Long)o;
	}
	
	@Override
	public WebDbModel getById(String cls, long id) throws Throwable
	{
		return GenericStorage.getObjectByIDDescendantOf(id, fetchClass(cls), srcGdb.get());
	}
	
	@Override
	public ValidationResult<TranslationFriendlyValidationEntry> update(String cls, DataObject obj) throws Exception
	{
		return RemoteGsdbTools.tryUpdateFrom
		(
			srcGdb.get(),
			fetchClass(cls),
			fetchId(obj),
			obj,
			"err_requested_element_not_exitst"
		);
	}
	
	@Override
	public ValidationResult<TranslationFriendlyValidationEntry> create(String cls, DataObject obj) throws Exception
	{
		return RemoteGsdbTools.createNewFrom
		(
			srcGdb.get(),
			fetchClass(cls),
			obj
		);
	}
	
	@Override
	public ValidationResult<TranslationFriendlyValidationEntry> delete(String cls, long id, DataObject deleteInstruction) throws Exception
	{
		return RemoteGsdbTools.delete
		(
			srcGdb.get(),
			fetchClass(cls),
			id,
			deleteInstruction
		);
	}
	
	//TODO limit offset, order
	@Override
	public List<WebDbModel> select(String cls, DataObject criteria, DataArray selectionExtra) throws Exception
	{
		LogicalGroup lg = RemoteGsdbTools.parseCriteria(criteria);
		ArrayList<WebDbModel> ret = new ArrayList<>();
		
		//TODO selectionExtra
		
		GenericStorage.getAllObjectsByQuery(fetchClass(cls), lg, ret, srcGdb.get());
		return ret;
	}

	@Override
	public Set<String> getPermissions()
	{
		return permissions.get();
	}

	public void fork(String name)
	{
		throw new RuntimeException("Forking database not supported");
	}

	@Ignore
	@Override
	public GenericStoreDatabase getBackendDatabase()
	{
		return srcGdb.get();
	}
}
