package eu.uartbus.webapp.modules;

import java.io.IOException;

import org.teavm.jso.dom.events.KeyboardEvent;
import org.teavm.jso.dom.html.HTMLElement;

import eu.javaexperience.electronic.uartbus.UartbusConsoleEnv;
import eu.javaexperience.electronic.uartbus.UartbusTools;
import eu.javaexperience.electronic.uartbus.rpc.client.UartBus;
import eu.javaexperience.electronic.uartbus.UartbusConsoleEnv.UbConsoleResponse;
import eu.javaexperience.interfaces.simple.publish.SimplePublish1;
import eu.javaexperience.patterns.behavioral.mediator.EventMediator;
import eu.javaexperience.reflect.Mirror;
import eu.jvx.js.lib.bindings.H;
import eu.jvx.js.lib.bindings.VanillaTools;

public class UbWebConsole
{
	protected H root;
	protected SimplePublish1<byte[]> sendPacket;
	protected H console = new H("div").attrs("style", "height: 300px;overflow: scroll;overflow-x: hidden;background-color: #eeeeee");
	protected H clear = new H("button").attrs("#text", "Clear").on("click", e->console.clear());
	protected H input = new H("input").attrs("type", "text").on("keyup", e->
	{
		KeyboardEvent event = (KeyboardEvent) e;
		if(event.getKeyCode() == 13)
		{
			doSend();
		}
	});
	
	protected boolean echo = true;
	protected H send = new H("button").attrs("#text", "Send").on("click", e->doSend());
	
	protected EventMediator<UbConsoleResponse> onAddressChanged = new EventMediator<>();
	
	protected UartbusConsoleEnv cmd = new UartbusConsoleEnv();
	
	public void reset()
	{
		cmd.setFromAddress(63);
		cmd.setToAddress(63);
		console.clear();
		VanillaTools.setContent(input, "");
	}
	
	public UartbusConsoleEnv getConsole()
	{
		return cmd;
	}
	
	public void incomingPacket(byte[] data)
	{
		addText((UartbusTools.isPacketCrc8Valid(data)?"":"!")+UartbusTools.formatColonData(data));
	}
	
	
	public void addText(String text)
	{
		addContent(new H("div").attrs("#text", text));
	}
	public void addContent(H content)
	{
		console.addChilds(content);
		HTMLElement out = console.getHtml();
		out.setScrollTop(out.getScrollHeight() - out.getClientHeight());
	}
	
	public void doSend()
	{
		String in = VanillaTools.getContent(input);
		VanillaTools.setContent(input, "");
		
		UbConsoleResponse ret = cmd.feed(in);
		
		switch (ret.getType())
		{
		case NO_OP:
			return;
		case NEED_HELP:
			printHelp();
			break;
		
		case QUERY_ADDRESS_FROM:
			addText("From address: "+cmd.getAddressFrom());
			onAddressChanged.dispatchEvent(ret);
			break;
		case QUERY_ADDRESS_TO:
			addText("To address: "+cmd.getAddressTo());
			onAddressChanged.dispatchEvent(ret);
			break;
		case SEND_PACKET:
			try
			{
				byte[] packet = ret.toPacket();
				System.out.println("send packet: "+UartbusTools.formatColonData(packet));
				if(echo)
				{
					addText("# "+UartbusTools.formatColonData(packet));
				}
				sendPacket.publish(packet);
			}
			catch (IOException e)
			{
				Mirror.propagateAnyway(e);
			}
			break;
		case UNKNOWN_COMMAND:
			unrecognisedCmd(ret.getMessage());
			break;
		default:
			break;
		}
	}
	
	public UbWebConsole(H root, SimplePublish1<byte[]> sendPacket)
	{
		this.root = root;
		this.sendPacket = sendPacket;
		root.addChilds
		(
			console,
			new H("div").addChilds
			(
				clear,
				input,
				send
			)
		);
		reset();
	}
	
	public static UbWebConsole connectToBus(H container, UartBus bus)
	{
		UbWebConsole ret = new UbWebConsole(container, p->bus.sendRawPacket(p));
		bus.getUnrelatedPacketListener().addEventListener(ret::incomingPacket);
		return ret;
	}
	
	protected void unrecognisedCmd(String message)
	{
		addText("Unrecognised command: "+message);
	}
	
	protected void printHelp()
	{
		StringBuilder sb = new StringBuilder();
		for(String s:UartbusConsoleEnv.HELP_LINES)
		{
			sb.append(s);
			sb.append("\n");
		}
		
		addContent(new H("pre").attrs("#text", sb.toString()));
	}
	
	public boolean isEchoEnabled()
	{
		return echo;
	}
	
	public void setEcho(boolean echo)
	{
		this.echo = echo;
	}
	
	public EventMediator<UbConsoleResponse> getOnAddressChange()
	{
		return onAddressChanged;
	}
}
