package eu.javaexperience.webtest;

import java.net.Inet4Address;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.Proxy.Type;

import org.openqa.selenium.WebDriver;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.firefox.FirefoxOptions;
import org.openqa.selenium.htmlunit.HtmlUnitDriver;

import eu.javaexperience.exceptions.UnimplementedCaseException;
import eu.javaexperience.parse.ParsePrimitive;
import eu.javaexperience.text.StringTools;

public class WebDriverSource
{
	protected WebDriverEnvConfig config;

	public WebDriverSource(WebDriverEnvConfig config)
	{
		this.config = config;
	}
	
	public enum AvailableWebDrivers
	{
		Chrome,
		Firefox,
		HtmlUnit
	};
	
	public static enum WebDriverOptions
	{
		Extension,
		Proxy,
	}
	
	public static class WebDriverConfiguration
	{
		public Proxy proxy;
		public boolean headless;
		
	}
	
	public WebDriver newDriver(AvailableWebDrivers drv, WebDriverConfiguration cfg)
	{
		switch(drv)
		{
		case Chrome:	return newChromeDriver(cfg);
		case Firefox:	return newFirefoxDriver(cfg);
		case HtmlUnit:	return newHtmlUnitDriver(cfg);
		default:
			throw new UnimplementedCaseException(drv);
		}
	}
	
	public WebDriver newFirefoxDriver(WebDriverConfiguration cfg)
	{
		System.setProperty("webdriver.gecko.driver", config.geckoDrivePath);
		System.setProperty("webdriver.firefox.bin", config.firefoxBinPath);
		
		FirefoxOptions profile = new FirefoxOptions();
		if(null != cfg)
		{
			if(null != cfg.proxy)
			{
				profile.setCapability("network.proxy.type", 1);
				String addr = cfg.proxy.address().toString();
				
				String host = StringTools.getSubstringAfterFirstString(StringTools.getSubstringBeforeFirstString(addr, ":"), "/");
				Object port = ParsePrimitive.tryParseInt(StringTools.getSubstringAfterLastString(addr, ":"));
				
				//if(Type.HTTP == cfg.proxy.type())
				{
					
					profile.setCapability("network.proxy.http", host);
					profile.setCapability("network.proxy.http_port", port);
				}
				//else if(Type.SOCKS == cfg.proxy.type())
				{
					profile.setCapability("network.proxy.socks", host);
					profile.setCapability("network.proxy.socks_port", port);
				}
			}
		}
		return new FirefoxDriver(profile);
	}
	
	public WebDriver newChromeDriver(WebDriverConfiguration cfg)
	{
		System.setProperty("webdriver.chrome.driver", config.chromeDrivePath);
		
		ChromeOptions options = new ChromeOptions();
		if(null != cfg)
		{
			if(null != cfg.proxy)
			{
				Proxy p = cfg.proxy;
				if(p.type() != Type.DIRECT)
				{
					options.addArguments("--proxy-server="+(p.type().toString().toLowerCase())+"://"+p.address());
				}
			}
			
			if(cfg.headless)
			{
				 options.addArguments("headless");
			}
		}
		return new ChromeDriver(options);
	}
	
	public WebDriver newHtmlUnitDriver(WebDriverConfiguration cfg)
	{
		return new HtmlUnitDriver(true);
	}
	
	public static void main(String[] args) throws Throwable
	{
		WebDriverEnvConfig env = new WebDriverEnvConfig();
		WebDriverSource src = new WebDriverSource(env);
		WebDriverConfiguration cfg = new WebDriverConfiguration();
		cfg.proxy = new Proxy(Type.SOCKS, new InetSocketAddress(Inet4Address.getByName("127.0.0.1"), 9050));
		
		src.newFirefoxDriver(cfg);
		
		while(true)
			Thread.sleep(50000);
	}
}
