package net.sf.snmpadaptor4j.core.trap;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import javax.management.Notification;
import net.sf.snmpadaptor4j.core.mapping.DataMapTrapMapping;
import net.sf.snmpadaptor4j.core.mapping.GenericSnmpTrapMapping;
import net.sf.snmpadaptor4j.core.mapping.MapDataMapTrapMapping;
import net.sf.snmpadaptor4j.core.mapping.SimpleDataMapTrapMapping;
import net.sf.snmpadaptor4j.core.mapping.SnmpTrapMapping;
import net.sf.snmpadaptor4j.core.mapping.SpecificSnmpTrapMapping;
import net.sf.snmpadaptor4j.core.mapping.UserDataEntryDataMapTrapMapping;
import net.sf.snmpadaptor4j.mbean.SystemInfo;
import net.sf.snmpadaptor4j.object.SnmpDataType;
import net.sf.snmpadaptor4j.object.SnmpOid;
import net.sf.snmpadaptor4j.object.SnmpTrap;
import net.sf.snmpadaptor4j.object.SnmpTrapData;

/**
 * Builder of SNMP traps from JMX notifications for an MBean.
 * @author <a href="http://fr.linkedin.com/in/jpminetti/">Jean-Philippe MINETTI</a>
 */
public class SnmpTrapBuilder
		implements Serializable {

	/**
	 * Serial number.
	 */
	private static final long serialVersionUID = -4410423782282390100L;

	/**
	 * Map of mappings to build SNMP traps for each notification type.
	 */
	private final Map<String, SnmpTrapMapping> mappingMap;

	/**
	 * Informations on the system.
	 */
	private final SystemInfo systemInfo;

	/**
	 * Constructor.
	 * @param mappingMap Map of mappings to build SNMP traps for each notification type.
	 * @param systemInfo Informations on the system.
	 */
	public SnmpTrapBuilder (final Map<String, SnmpTrapMapping> mappingMap, final SystemInfo systemInfo) {
		super();
		this.mappingMap = mappingMap;
		this.systemInfo = systemInfo;
	}

	/**
	 * Returns the map of mappings to build SNMP traps for each notification type.
	 * @return Map of mappings to build SNMP traps for each notification type.
	 */
	public Map<String, SnmpTrapMapping> getMappingMap () {
		return this.mappingMap;
	}

	/**
	 * Creates and returns a new SNMP trap by a JMX notification.
	 * @param notification JMX notification generated by a MBean.
	 * @return New SNMP trap or <code>NULL</code> if the notification is not mapped.
	 */
	public SnmpTrap newTrap (final Notification notification) {
		SnmpTrap trap = null;
		final SnmpTrapMapping mapping = this.mappingMap.get(notification.getType());
		if (mapping instanceof GenericSnmpTrapMapping) {
			final Map<SnmpOid, SnmpTrapData> dataMap = newDataMap(mapping.getDataMap(), notification);
			trap = SnmpTrap.newInstance(notification.getTimeStamp() / 10, mapping.getSource(), ((GenericSnmpTrapMapping) mapping).getType(), dataMap);
		}
		else if (mapping instanceof SpecificSnmpTrapMapping) {
			final Map<SnmpOid, SnmpTrapData> dataMap = newDataMap(mapping.getDataMap(), notification);
			trap = SnmpTrap.newInstance(notification.getTimeStamp() / 10, mapping.getSource(), ((SpecificSnmpTrapMapping) mapping).getType(), dataMap);
		}
		return trap;
	}

	/**
	 * Creates and returns a new <code>dataMap</code> of {@link SnmpTrap} by a JMX notification.
	 * @param mapping Mapping to build the <code>dataMap</code> field of {@link SnmpTrap} from a JMX notification.
	 * @param notification JMX notification generated by a MBean.
	 * @return New <code>dataMap</code> of {@link SnmpTrap}.
	 */
	protected final Map<SnmpOid, SnmpTrapData> newDataMap (final DataMapTrapMapping mapping, final Notification notification) {
		final Map<SnmpOid, SnmpTrapData> dataMap = new HashMap<SnmpOid, SnmpTrapData>();
		if (mapping.getSequenceNumberOid() != null) {
			dataMap.put(mapping.getSequenceNumberOid(), new SnmpTrapData(SnmpDataType.unsigned32, new Long(notification.getSequenceNumber())));
		}
		if (mapping.getMessageOid() != null) {
			dataMap.put(mapping.getMessageOid(), new SnmpTrapData(SnmpDataType.octetString, notification.getMessage()));
		}
		if (mapping.isHasSystemInfo() && (this.systemInfo != null)) {
			dataMap.put(SnmpOid.SYSNAME_OID, new SnmpTrapData(SnmpDataType.octetString, this.systemInfo.getSysName()));
			dataMap.put(SnmpOid.SYSDESCR_OID, new SnmpTrapData(SnmpDataType.octetString, this.systemInfo.getSysDescr()));
			dataMap.put(SnmpOid.SYSLOCATION_OID, new SnmpTrapData(SnmpDataType.octetString, this.systemInfo.getSysLocation()));
			dataMap.put(SnmpOid.SYSCONTACT_OID, new SnmpTrapData(SnmpDataType.octetString, this.systemInfo.getSysContact()));
		}
		if (mapping instanceof SimpleDataMapTrapMapping) {
			final SimpleDataMapTrapMapping simpleMapping = (SimpleDataMapTrapMapping) mapping;
			if (notification.getUserData() instanceof Map) {
				throw new RuntimeException("UserData field of notification \"" + notification.getType() + "\" contains map - see mapping file of MBean");
			}
			dataMap.put(simpleMapping.getUserDataOid(), new SnmpTrapData(simpleMapping.getUserDataType(), notification.getUserData()));
		}
		else if (mapping instanceof MapDataMapTrapMapping) {
			if (notification.getUserData() != null) {
				final MapDataMapTrapMapping mapMapping = (MapDataMapTrapMapping) mapping;
				if (!(notification.getUserData() instanceof Map)) {
					throw new RuntimeException("UserData field of notification \"" + notification.getType() + "\" doesn't contain map - see mapping file of MBean");
				}
				final Map<?, ?> userDataMap = (Map<?, ?>) notification.getUserData();
				for (final UserDataEntryDataMapTrapMapping userDataEntry : mapMapping.getUserDataEntryList()) {
					if (userDataMap.containsKey(userDataEntry.getKey())) {
						dataMap.put(userDataEntry.getOid(), new SnmpTrapData(userDataEntry.getType(), userDataMap.get(userDataEntry.getKey())));
					}
				}
			}
		}
		return dataMap;
	}

	/*
	 * {@inheritDoc}
	 * @see java.lang.Object#toString()
	 */
	@Override
	public final String toString () {
		return "SnmpTrapBuilder:" + this.mappingMap;
	}

}